/*
	The Controller handles mushroom nursery maintenance.
*/

#include "Arduino.h"
#include "Controller.h"
#include "Utils.h"
#include "ArduinoJson.h"
#include "Internet.h"
#include "Preferences.h" //Storage for NVS see: https://github.com/espressif/arduino-esp32/tree/master/libraries/Preferences
#include "SparkFun_RHT03.h"
#include "NTPClient.h"
#include "WiFiUdp.h"

Preferences prefs;
RHT03 rht;
WiFiUDP ntpUDP;
NTPClient timeClient(ntpUDP, -18000);


const int SENSOR_A_PIN = 19; //Humidity/Temp Sensor
const int DEVICE_A_PIN = 21; //IntakeFan(old) --> UV Light
const int DEVICE_B_PIN = 16; //Atomizer(old) --> Day Light
const int DEVICE_C_PIN = 32; //ExhaustFan(old) --> ExhaustFan
const int DEVICE_D_PIN = 27; //Daylight(old) --> Intake Fan
const int DEVICE_E_PIN = 14; //UVLight(old) --> Atomizer

Controller::Controller(int test){}


void Controller::init(){
	Serial.println("________________________________________________");
	Serial.println("Controller starting, and fetching config from prefs.");

	//Setup of pins.
	rht.begin(SENSOR_A_PIN);
    pinMode(DEVICE_A_PIN, OUTPUT);
    pinMode(DEVICE_B_PIN, OUTPUT);
    pinMode(DEVICE_C_PIN, OUTPUT);
    pinMode(DEVICE_D_PIN, OUTPUT);
    pinMode(DEVICE_E_PIN, OUTPUT);


    //Setup Exhaust Timer
    _cycleStartTime = 0;
    _cycleON = false;



	_configReady = false;

	//Load stored configuration.
	prefs.begin("config-data", true);
	String configString = prefs.getString("config", "");
	prefs.end();

	if(configString == ""){
		Serial.println("Stored config is invalid, did not apply it.");
		return;
	}

	JsonDocument config = Utils::parseJSON(configString);
	
	applyConfig(config);

	readSensor();

	manageDevices();

}

void Controller::readSensor(){

	int updateRet = rht.update();

	  // 1 indicates update was successful.
	  if (updateRet == 1){

	    // float temp_c = rht.tempC();
	    _tempF 		= rht.tempF();
	    _humidity 	= rht.humidity();

	    Serial.print(_tempF);
	    Serial.print("F ");
	    Serial.print(_humidity);
	    Serial.println("%");

	  }


}


void Controller::applyConfig(JsonDocument config){

	//Exhaust Fan values
	_exhaustFanOn 				= config["exhaustFanOn"];
	_exhaustFanOverrideEnable 	= config["exhaustFanOverrideEnable"];
	_exhaustFanOnCycle 			= config["exhaustFanOnCycle"];
	_exhaustFanOffCycle 		= config["exhaustFanOffCycle"];

	//Humidfier
	_humidiferOverrideEnable 	= config["humidifierOverrideEnable"];
	_atomizerOn					= config["atomizerOn"];
	_intakeFanOn				= config["intakeFanOn"];
	_humiditySetValue			= config["humiditySetValue"];
	_humidityDiffValue			= config["humidityDiffValue"];

	//Daylight
	_dayLightOn 				= config["dayLightOn"];
	_dayLightOverrideEnable 	= config["dayLightOverrideEnable"];
	_dayLightStartHour			= config["dayLightStartHour"];
	_dayLightStartMin			= config["dayLightStartMin"];
	_dayLightEndHour			= config["dayLightEndHour"];
	_dayLightEndMin				= config["dayLightEndMin"];

	//UVlight
	_uvLightOn 					= config["uvLightOn"];
	_uvLightOverrideEnable 		= config["uvLightOverrideEnable"];
	_uvLightStartHour			= config["uvLightStartHour"];
	_uvLightStartMin			= config["uvLightStartMin"];
	_uvLightEndHour				= config["uvLightEndHour"];
	_uvLightEndMin				= config["uvLightEndMin"];


	_configReady = true;

	Serial.println("Config applied and ready.");

}


void Controller::loop(Internet internet){

	Serial.println("Controller Loop here. Fetching new config...");
	
	if(internet.isConnected() == true){

		timeClient.update();
		Serial.println(timeClient.getFormattedTime());

		
		//Build request string
		String postData = 	String("{") + 
								"\"intakeFanOn\":" + Utils::bTs(_STATUS_INTAKE_ON) + "," + 
								"\"exhaustFanOn\":" + Utils::bTs(_STATUS_EXHAUST_ON) + "," +
								"\"atomizerOn\":" + Utils::bTs(_STATUS_ATOMIZER_ON) + "," +
								"\"dayLightOn\":" + Utils::bTs(_STATUS_DAYLIGHT_ON) + "," +
								"\"uvLightOn\":" + Utils::bTs(_STATUS_UVLIGHT_ON) + "," +
								"\"temp\":" + String( (int) _tempF ) + "," +
								"\"exhaustFanOnCycle\":" + _exhaustFanOnCycle + "," +
								"\"exhaustFanOffCycle\":" + _exhaustFanOffCycle + "," +
								"\"dayLightStartHour\":" + _dayLightStartHour + "," +
								"\"dayLightStartMin\":" + _dayLightStartMin + "," +
								"\"dayLightEndHour\":" + _dayLightEndHour + "," +
								"\"dayLightEndMin\":" + _dayLightEndMin + "," +
								"\"uvLightStartHour\":" + _uvLightStartHour + "," +
								"\"uvLightStartMin\":" + _uvLightStartMin + "," +
								"\"uvLightEndHour\":" + _uvLightEndHour + "," +
								"\"uvLightEndMin\":" + _uvLightEndMin + "," +
								"\"humidity\":" + String( (int) _humidity ) + //Last line
							"}"; 






		String response = internet.checkInRequest(postData);


		//If config is good we save it and apply it.
		if(response != "" && response != "-1"){
			//Save config in prefs
			prefs.begin("config-data", false);
			prefs.putString("config", response);
			prefs.end();

			//Apply new config
			JsonDocument config = Utils::parseJSON(response);
			applyConfig(config);
		}
		else{
			Serial.println("Fetched config from web but response is invalid did not save or apply it.");
		}


	}
	else{
		Serial.println("No internet connection, skipping fetch of time and checkin.");
	}


	//Read Sensor
	readSensor();

	//Manage Devices
	manageDevices();

}


void Controller::manageDevices(){

	if(_configReady == false){
		Serial.println("Config not ready, skipping device management.");
		return;
	}

	manageExhaustFan();
	manageHumidifer();
	manageDayLight();
	manageUvLight();



}


void Controller::manageUvLight(){
	// "uvLightOn": false,
	// "uvLightOverrideEnable": false,
	// "uvLightStart": "ISO",
	// "uvLightEnd": "ISO",
	// "uvLightStartHour": 10,
 	// "uvLightStartMin": 0,
 	// "uvLightEndHour": 10,
 	// "uvLightEndMin": 0,


	if(_uvLightOverrideEnable == true){

		//Manual

		if(_uvLightOn == true){
			digitalWrite(DEVICE_A_PIN, HIGH);
			_STATUS_UVLIGHT_ON = true;
		}
		else{
			digitalWrite(DEVICE_A_PIN, LOW);
			_STATUS_UVLIGHT_ON = false;	
		}

	}
	else{

		//Autopilot
		int currentHours = timeClient.getHours();
		int currentMinutes = timeClient.getMinutes();


		bool shouldEnable = Utils::enableIfWithinTimeRange(currentHours, currentMinutes, _uvLightStartHour, _uvLightStartMin, _uvLightEndHour, _uvLightEndMin);

		if(shouldEnable == true){
			digitalWrite(DEVICE_A_PIN, HIGH);
			_STATUS_UVLIGHT_ON = true;
		}
		else{
			digitalWrite(DEVICE_A_PIN, LOW);
			_STATUS_UVLIGHT_ON = false;	
		}


	}

	Serial.println(" -- UVLight --");
	Serial.println("   UVLIGHT_ON: " +  String(_STATUS_UVLIGHT_ON));
	Serial.println("   uvLightOn: " + String(_uvLightOn) + "- uvLightOverrideEnable: " + _uvLightOverrideEnable + " - uvLightStart: " + _uvLightStartHour + ":" + _uvLightStartMin + " - uvLightEnd: " + _uvLightEndHour + ":" + _uvLightEndMin);


}


void Controller::manageDayLight(){
	// "dayLightOn": false,
	// "dayLightOverrideEnable": false,
	// "dayLightStartHour": 10,
 	// "dayLightStartMin": 0,
 	// "dayLightEndHour": 10,
 	// "dayLightEndMin": 0,


	if(_dayLightOverrideEnable == true){

		//Manual

		if(_dayLightOn == true){
			digitalWrite(DEVICE_B_PIN, HIGH);
			_STATUS_DAYLIGHT_ON = true;
		}
		else{
			digitalWrite(DEVICE_B_PIN, LOW);
			_STATUS_DAYLIGHT_ON = false;	
		}

	}
	else{

		//Autopilot
		int currentHours = timeClient.getHours();
		int currentMinutes = timeClient.getMinutes();


		bool shouldEnable = Utils::enableIfWithinTimeRange(currentHours, currentMinutes, _dayLightStartHour, _dayLightStartMin, _dayLightEndHour, _dayLightEndMin);

		if(shouldEnable == true){
			digitalWrite(DEVICE_B_PIN, HIGH);
			_STATUS_DAYLIGHT_ON = true;
		}
		else{
			digitalWrite(DEVICE_B_PIN, LOW);
			_STATUS_DAYLIGHT_ON = false;
		}


	}

	Serial.println(" -- Daylight --");
	Serial.println("   DAYLIGHT_ON: " +  String(_STATUS_DAYLIGHT_ON));
	Serial.println("   dayLightOn: " + String(_dayLightOn) + "- dayLightOverrideEnable: " + _dayLightOverrideEnable + " - dayLightStart: " + _dayLightStartHour + ":" + _dayLightStartMin + " - dayLightEnd: " + _dayLightEndHour + ":" + _dayLightEndMin);


}

void Controller::manageHumidifer(){

	//Related values
	// "humiditySetValue":95,
	// "humidityDiffValue": 3,
	// "intakeFanOn": true,
	// "atomizerOn": false,
	// "humidiferOverrideEnable":false,


	if(_humidiferOverrideEnable == true){
		//We are in manual override mode.

		if(_intakeFanOn == true){
			digitalWrite(DEVICE_D_PIN, HIGH);
			_STATUS_INTAKE_ON = true;
		}
		else{
			digitalWrite(DEVICE_D_PIN, LOW);
			_STATUS_INTAKE_ON = false;
		}

		if(_atomizerOn == true){
			digitalWrite(DEVICE_E_PIN, HIGH);
			_STATUS_ATOMIZER_ON = true;
		}
		else{
			digitalWrite(DEVICE_E_PIN, LOW);
			_STATUS_ATOMIZER_ON = false;
		}

	}
	else{

		//We are in autopilot mode.
		int humidity = (int) _humidity;

		//If humidity is less than the kickon value, enable the humidifer.
		if( humidity < (_humiditySetValue - _humidityDiffValue) ){
			
			//If atomizer was previously off then delay fan by 30 sec.
			if(_STATUS_ATOMIZER_ON == false){
				_ATOMIZER_START_TIME = millis();
			}

			//delay exhaust fan for 60 seconds to create fog layer.
			if( millis() - _ATOMIZER_START_TIME >= 60000 ){
				digitalWrite(DEVICE_D_PIN, HIGH);
				_STATUS_INTAKE_ON = true;

			}

			Serial.println("Atomizer running for " + String(millis() - _ATOMIZER_START_TIME) + "ms");

			digitalWrite(DEVICE_E_PIN, HIGH);
			_STATUS_ATOMIZER_ON = true;


		}
		//If humidity is at the desired level or greater, turn off humidifer.
		else if( humidity >= _humiditySetValue){
			digitalWrite(DEVICE_D_PIN, LOW);
			digitalWrite(DEVICE_E_PIN, LOW);
			_STATUS_INTAKE_ON = false;
			_STATUS_ATOMIZER_ON = false;
		}

	}

	Serial.println(" -- Humidifier --");
	Serial.println("   INTAKE_ON: " +  String(_STATUS_INTAKE_ON));
	Serial.println("   ATOMIZER_ON: " +  String(_STATUS_ATOMIZER_ON));
	Serial.println("   IntakeFanOn: " + String(_intakeFanOn) + "- atomizerOn: " + _atomizerOn + " - humidiferOverride: " + _humidiferOverrideEnable + " - humiditySetValue: " + _humiditySetValue + " - humidityDiffValue: " + _humidityDiffValue);

}


void Controller::manageExhaustFan(){

	//Use the below & sensor data to determine if enable or disable fan.
	// _exhaustFanOn 				
	// _exhaustFanOverrideEnable
	// _exhaustFanOnCycle 1min
	// _exhaustFanOffCycle 9min
	Serial.println(" -- Exhaust --");


	if( _exhaustFanOverrideEnable == true ){

		//Manual mode

		if( _exhaustFanOn == true ){
			digitalWrite(DEVICE_C_PIN, HIGH);
			_STATUS_EXHAUST_ON = true;
		}
		else{
			digitalWrite(DEVICE_C_PIN, LOW);
			_STATUS_EXHAUST_ON = false;
		}

	}
	else{
		//Autopilot mode

		unsigned long int now = millis();
		unsigned long int cycleDuration = 0;
		unsigned long int timeElapsed = now - _cycleStartTime;

		// Serial.println("Time Elapsed: " + String(timeElapsed) + "ms");

		//Calculate the current cycle duration
		if(_cycleON == true){
			//calculate on duration in ms
			cycleDuration = _exhaustFanOnCycle * 60000;
		}
		else{
			//Calculate off duration in ms
			cycleDuration = _exhaustFanOffCycle * 60000;
		}

		// Serial.println("Cycle Duration: " + String(cycleDuration) + "ms");

		if( _cycleStartTime == 0 ){
			//Board was just turned on, start Off cycle.
			_cycleON = false;
			_cycleStartTime = millis();
			digitalWrite(DEVICE_C_PIN, LOW);
			_STATUS_EXHAUST_ON = false;
		}
		else if( timeElapsed >= cycleDuration){
			//Interval time is up, switch mode.
			_cycleStartTime = millis();

			if(_cycleON == true){
				//Switch to off mode.
				digitalWrite(DEVICE_C_PIN, LOW);
				_STATUS_EXHAUST_ON = false;
				_cycleON = false;
			}
			else{
				//Switch to on mode.
				digitalWrite(DEVICE_C_PIN, HIGH);
				_STATUS_EXHAUST_ON = true;
				_cycleON = true;
			}

		}

		Serial.println("   Time Elapsed: " + String(timeElapsed) + "ms" + " - Cycle Duration: " + String(cycleDuration) + "ms");

	}

	
	Serial.println("   EXHAUST_ON: " +  String(_STATUS_EXHAUST_ON));
	Serial.println("   ExhaustFanOn: " + String(_exhaustFanOn) + " - Override: " + _exhaustFanOverrideEnable + " - OnCycle: " + _exhaustFanOnCycle + " - OffCycle: " + _exhaustFanOffCycle);



}




